﻿/**********************************************************************************
 *
 * File:            Program.cs
 *
 * Usage:           AddMetadata <inputPath> <outputPath> [<mdataFile>]
 *                  Example: in.pdf out.pdf MetadataTest.pdf
 *                  
 * Title:           Add metadata to PDF
 *                  
 * Description:     Set metadata such as author, title, and creator of a PDF
 *                  document. Optionally use the metadata of another PDF
 *                  document or the content of an XMP file.
 *                  
 * Author:          株式会社トラスト・ソフトウェア・システム
 *
 * Copyright:       Copyright (C) 2025 株式会社トラスト・ソフトウェア・システム
 *
 *********************************************************************************/

using System;
using System.IO;
using PdfTools;
using PdfTools.Pdf;
using PdfTools.Optimization;
using PdfTools.Optimization.Profiles;
using System.Net.NetworkInformation;

namespace AddMetadata
{
    internal class Program
    {
        static void Usage()
        {
            Console.WriteLine("Usage: AddMetadata <inputPath> <outputPath> [<mdataFile>]");
        }

        static void Main(string[] args)
        {
            // 引数チェック
            if (args.Length < 2 || args.Length > 3)
            {
                Usage();
                return;
            }

            try
            {
                // 既定ではテストライセンスとしての利用が可能ですが、結果には「透かし」が追加されます。
                // ライセンスキーをお持ちの場合は以下のコメントを削除し、そのライセンスキーをセットしてください。
                // Sdk.Initialize("ここにライセンスキーをセット");

                AddMetadata(args[0], args[1], args.Length == 3 ? args[2] : "");
                Console.WriteLine("Execution successful.");
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
            }
        }

        private static void AddMetadata(string inPath, string outPath, string mDataFile)
        {
            //入力ファイル
            using var inStream = File.OpenRead(inPath);
            var inDoc = Document.Open(inStream);

            //メタデータ作成
            var metaSet = new MetadataSettings();

            try
            {
                using var mStr = File.OpenRead(mDataFile);
                var mDoc = Document.Open(mStr);
                var meta = mDoc.Metadata;
                metaSet.Author = meta.Author == "" ? "" : meta.Author;
                metaSet.CreationDate = meta.CreationDate == null ? null : meta.CreationDate;
                metaSet.Creator = meta.Creator == "" ? meta.Creator : "";
                metaSet.ModificationDate = meta.ModificationDate == null ? null : meta.ModificationDate;
                metaSet.Producer = meta.Producer == null ? null : meta.Producer;
                metaSet.Subject = meta.Subject == null ? null : meta.Subject;
                metaSet.Title = meta.Title == null ? null : meta.Title;
            }
            catch (Exception)
            {
                metaSet.Author = "someAuther";
                metaSet.CreationDate = DateTimeOffset.Now;
                metaSet.Creator = "someCreator";
                metaSet.ModificationDate = DateTimeOffset.Now;
                metaSet.Producer = "SomeProducer";
                metaSet.Subject = "Subject";
                metaSet.Title = "Title";
            }

            //出力オプション
            var outOpt = new OutputOptions();
            outOpt.MetadataSettings = metaSet;

            //Profile
            var profile = new MinimalFileSize();

            //出力Stream作成
            using var outStr = File.Create(outPath);

            //出力
            using var outDoc = new Optimizer().OptimizeDocument(inDoc, outStr, profile, outOpt);
        }
    }
}
